package ga.view.streaming.showroom;

import com.jme3.math.Vector3f;
import com.jme3.renderer.Camera;
import com.jme3.system.AppSettings;

/**
 * This contains the settings needed to configure a cameras position and
 * perspective.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
/**
 * 
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class CameraSettings {
  /**
   * Types of perspectives.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public enum Type {

    /** The isometric perspective from above. */
    ISOMETRIC,
    /** The orthographic aerial perspective. */
    ORTHOGRAPHIC,
    /** The in room perspective. */
    IN_ROOM
  }

  private final Vector3f initialLocation;

  private Vector3f lookAtPoint = Vector3f.ZERO;

  private final boolean isIsometric;

  private float minVRotation = 0.2f;
  private float maxVRotation = 0.3f;
  private float minDistance = 1f;
  private float maxDistance = 10f;

  private float frustumNear = 0.1f;
  private float frustumFar = 20f;

  private final float fov;

  private final float aspect;

  /**
   * Instantiates a new camera settings.
   * 
   * @param initialLocation
   *          the initial location
   * @param lookAtPoint
   *          the look at point
   * @param minVRotation
   *          the min v rotation
   * @param maxVRotation
   *          the max v rotation
   * @param minDistance
   *          the min distance
   * @param maxDistance
   *          the max distance
   * @param aspect
   *          the aspect
   * @param fov
   *          the fov
   * @param frustumNear
   *          the frustum near
   * @param frustumFar
   *          the frustum far
   * @param isIsometric
   *          the is isometric
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public CameraSettings(final Vector3f initialLocation,
      final Vector3f lookAtPoint, final float minVRotation,
      final float maxVRotation, final float minDistance,
      final float maxDistance, final float aspect, final float fov,
      final float frustumNear, final float frustumFar, final boolean isIsometric) {
    super();
    this.initialLocation = initialLocation;
    this.lookAtPoint = lookAtPoint;
    this.isIsometric = isIsometric;
    this.minVRotation = minVRotation;
    this.maxVRotation = maxVRotation;
    this.minDistance = minDistance;
    this.maxDistance = maxDistance;
    this.aspect = aspect;
    this.fov = fov;
    this.frustumNear = frustumNear;
    this.frustumFar = frustumFar;
  }

  /**
   * Configures a camera.
   * 
   * @param cam
   *          the camera
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void configureCamera(final Camera cam) {
    cam.setParallelProjection(isIsometric);
    cam.setLocation(initialLocation);
    cam.lookAt(lookAtPoint, Vector3f.UNIT_Y);
    cam.setFrustumNear(frustumNear);
    cam.setFrustumFar(frustumFar);

    if (isIsometric) {
      // offscreenCam.setFrustumPerspective(45f, (float)
      // vp.getCamera().getWidth()
      // / vp.getCamera().getHeight(), 0.1f, 20f);

      cam.setFrustum(frustumNear, frustumFar, aspect * -4f, aspect * 4f, 4f,
          -4f);
    } else {
      cam.setFrustumPerspective(fov, aspect, frustumNear, frustumFar);
    }

  }

  /**
   * Gets the look at point.
   * 
   * @return the look at point
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public Vector3f getLookAtPoint() {
    return lookAtPoint;
  }

  /**
   * Gets the max vertical rotation.
   * 
   * @return the max vertical rotation
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public float getMaxVRotation() {
    return maxVRotation;
  }

  /**
   * Gets the minimum distance.
   * 
   * @return the minimum distance
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public float getMinDistance() {
    return minDistance;
  }

  /**
   * Gets the maximum distance.
   * 
   * @return the maximum distance
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public float getMaxDistance() {
    return maxDistance;
  }

  /**
   * Gets the minimum vertical rotation.
   * 
   * @return the minimum vertical rotation
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public float getMinVRotation() {
    return minVRotation;
  }

  /**
   * Gets the settings for a show room and a type of perspective.
   * 
   * @param type
   *          the type of perspective.
   * @param room
   *          the show room
   * @param settings
   *          the app settings
   * @return the camera settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static CameraSettings getSettings(final Type type,
      final ShowRoom room, final AppSettings settings) {
    switch (type) {
    case IN_ROOM:
      return getInRoomSettings(room, settings);
    case ISOMETRIC:
      return getIsometricSettings(room, settings);
    case ORTHOGRAPHIC:
      return getOrthographicSettings(room, settings);
    }

    return null;
  }

  /**
   * Gets the in room perspective settings for a show room.
   * 
   * @param room
   *          the show room
   * @param settings
   *          the app settings
   * @return the camera settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static CameraSettings getInRoomSettings(final ShowRoom room,
      final AppSettings settings) {
    final Vector3f location = new Vector3f(3.0f, 1.8f, 3.0f);
    final Vector3f target = room.getLocalTranslation();
    final float minDistance = location.distance(target);
    final float maxDistance = minDistance * 1.5f;
    final float frustumFar = maxDistance * 4f;

    return new CameraSettings(location, target, 0.2f, 0.3f, minDistance,
        maxDistance, settings.getWidth() / (float) settings.getHeight(), 45f,
        0.1f, frustumFar, false);
  }

  /**
   * Gets the isometric perspective settings for a show room.
   * 
   * @param room
   *          the show room
   * @param settings
   *          the app settings
   * @return the camera settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static CameraSettings getIsometricSettings(final ShowRoom room,
      final AppSettings settings) {
    final Vector3f location = new Vector3f(70f, 60f, 70f);
    final Vector3f target = Vector3f.ZERO;
    final float maxDistance = location.distance(target) * 1.05f;
    final float frustumFar = maxDistance * 2f;

    return new CameraSettings(location, target, 0.2f, 0.9f, 1f, maxDistance,
        settings.getWidth() / (float) settings.getHeight(), 45f, 0.1f,
        frustumFar, true);
  }

  /**
   * Gets the orthographic perspective settings for a show room.
   * 
   * @param room
   *          the show room
   * @param settings
   *          the app settings
   * @return the camera settings
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static CameraSettings getOrthographicSettings(final ShowRoom room,
      final AppSettings settings) {
    final Vector3f location = new Vector3f(7f, 6f, 7f);
    final Vector3f target = new Vector3f(0f, -0.5f, 0f);
    final float minDistance = location.distance(target) * 0.5f;
    final float maxDistance = location.distance(target) * 1.05f;
    final float frustumFar = maxDistance * 2f;

    return new CameraSettings(location, target, 0.2f, 0.9f, minDistance,
        maxDistance, settings.getWidth() / (float) settings.getHeight(), 45f,
        0.1f, frustumFar, false);
  }

}
